///////////////////////////////////
// ********************************
// LENS-FLARE(S) 2013 by priest
//
// LENS-FLARE(S) 2020 - Translated in lite c  by Ayaka
//
// ****************************************************
///////////////////////////////////////////////////////





#include "CustomLensflares.h"








///
// Helper
// ------------------------
// 
/// 

///
// Create a new material with the given specifications
//
// <pow> Power (1-10)
// <spec> Specular rgb as an array
// <diff> Diffuse rgb as an array
// <amb> Ambient rgb as an array
MATERIAL* LfpMatCreate(int pow, var* spec, var* diff, var* amb)
{
	MATERIAL* mat = mtl_create();
	
	mat.power = pow;
	
	if(spec) { vec_set(mat.specular_blue, spec); }
	if(diff) { vec_set(mat.diffuse_blue, diff); }	
	if(amb)  { vec_set(mat.ambient_blue, amb); }
	
	return mat;
}







///
// Initialization & Cleanup
// ------------------------
// 
/// 

///
// Returns the ID associated with string1 as a number.
//
// <ent> The model with the assigned group
// <string1> Main group
// <string2> Sub group
// <isSubGrp> true, if the subset should be returned, otherwise false
int LfpGetGrpId(ENTITY* ent, char* string1, char* string2, bool isSubGrp)
{
	if(!isSubGrp)
	{
		if(str_cmp(string1, "LfpLTypeLong")) return LfpLTypeLong;
		if(str_cmp(string1, "LfpLTypePoint")) return LfpLTypePoint;
		if(str_cmp(string1, "LfpLTypeWindow")) return LfpLTypeWindow;
	}
	else
	{
		if(str_cmp(string2, "LfpNeonPipe1")) return LfpNeonPipe1;
		if(str_cmp(string2, "LfpNeonPipe2")) return LfpNeonPipe2;	
		
		if(str_cmp(string2, "LfpSun1")) return LfpSun1;
		if(str_cmp(string2, "LfpSun2")) return LfpSun2;
		if(str_cmp(string2, "LfpLamp1")) return LfpLamp1;

		if(str_cmp(string2, "LfpWindowRed")) return LfpWindowRed;
		if(str_cmp(string2, "LfpWindowWhite")) return LfpWindowWhite;		
	}

	return -1;
}

///
// Returns the ID associated with string1 as text.
//
// <id> Subgroup ID or main group ID
// <isSubGrp> true if the subset should be returned, otherwise false
char* LfpGetGrpText(int id, bool isSubGrp)
{
	if(!isSubGrp)
	{
		switch(id)
		{
			case LfpLTypeLong: return "LfpLTypeLong";
			case LfpLTypePoint: return "LfpLTypePoint";
			case LfpLTypeWindow: return "LfpLTypeWindow";				
		}		
	}
	else
	{
		switch(id)
		{
			case LfpNeonPipe1: return "LfpNeonPipe1";
			case LfpNeonPipe2: return "LfpNeonPipe2";	
			case LfpSun1: return "LfpSun1";	
			case LfpSun2: return "LfpSun2";
			case LfpLamp1: return "LfpLamp1";		
			case LfpWindowRed: return "LfpWindowRed";
			case LfpWindowWhite: return "LfpWindowWhite";									
		}		
	}	
	
	return NULL;
}

///
// Checks whether an ID is included in the list of groups and returns the ID if contained, otherwise -1
//
// <lfpGroups> List with groups
// <count> Number of (main) groups in the list
// <id> The id to search for
int LfpGrpExist(LfpGroup* lfpGrps, int count, int id)
{
	if(lfpGrps == NULL) { return -1; }
	
	int i;

	for(i=0; i < count; i++)
	{
		if(lfpGrps[i].Id == id)
		return i;
	}
	
	return -1;	
}

///
// Checks whether an ID is included in the list of groups and returns the ID if contained, otherwise -1
//
// <lfpGroups> List with groups
// <index> Index of a main group
// <id> The subgroups to be found Id
int LfpSubGrpExist(LfpGroup* lfpGrps, int index, int id)
{
	if(index < 0 || lfpGrps[index].Subs == NULL)
	{
		return -1;
	}

	int j;	
	int subCount = lfpGrps[index].Count;

	for(j=0; j < subCount; j++)
	{
		if(((lfpGrps[index]).Subs)[j].Id == id)
		{
			return j;
		}
	}
	
	return -1;	
}



///
// Add a single light
//
// <et> light model
// <mainGroup> Main group
// <mainGroup> Sub group
// <vertexNum> Vertex on parent model or NULL
void LfpAddLight(ENTITY* et, char* mainGroup, char* subGroup, var vertexNum)
{
	while(Lfp == NULL) { wait(10); }
	
	static int grpIdx = -1;
	static int subIdx = -1;
	int liSub = 0;
	int id= -1;
	int sId= -1;
	
	/* Get ID from top and bottom group */		
	id = LfpGetGrpId(et, mainGroup, subGroup, false);
	if(id == -1)
	{
		DbgWriteLog("\nThe name of string1 %s is not a valid Lfp name.", mainGroup);
		return;
	}
	
	sId = LfpGetGrpId(et, mainGroup, subGroup, true);
	if(sId == -1)
	{
		DbgWriteLog("\nThe name of string2 %s is not a valid Lfp name", subGroup);
		return;
	}
	

	
	
	/* Create groups */	
	grpIdx = LfpGrpExist(Lfp.Groups, Lfp.GroupCount, id);		
	if(grpIdx < 0)
	{
		grpIdx = Lfp.GroupCount;
		
		Lfp.Groups = realloc(Lfp.Groups, sizeof(LfpGroup) * (Lfp.GroupCount+1));
		LfpGroup* grp = &(Lfp.Groups)[grpIdx];		
		memset(grp,0,sizeof(LfpGroup));
		grp.Id = id;
		grp.Name = mainGroup;			
		Lfp.GroupCount++;
		
		DbgWriteLog("\nLfp group %d created.", grpIdx);	
	}

	/* Create subgroups */	
	subIdx = LfpSubGrpExist(Lfp.Groups, grpIdx ,sId);
	if(subIdx < 0)
	{
		LfpGroup* grp = &(Lfp.Groups)[grpIdx];	
		subIdx = grp.Count;
		
		grp.Subs = realloc(grp.Subs,sizeof(LfpSubGroup) * (subIdx+1));
		memset((grp.Subs)[subIdx],0,sizeof(LfpSubGroup));			
		(grp.Subs)[subIdx].Id = sId;
		(grp.Subs)[subIdx].Name = subGroup;
		grp.Count++;			
		
		DbgWriteLog("\n   Lfp subgroup %d created.", subIdx);
	}
	

	/* Generate light */
	LfpGroup* grp = &(Lfp.Groups)[grpIdx];	
	
	liSub = (grp.Subs)[subIdx].Count;
	(grp.Subs)[subIdx].Lights = realloc((grp.Subs)[subIdx].Lights, sizeof(LfpLight) * (liSub+1));
	
	LfpLight* light= &(((grp.Subs)[subIdx]).Lights)[liSub];
	memset(light,0,sizeof(LfpLight));

	(grp.Subs)[subIdx].Count++;
	if(!et.LfpOrderId)
	{
		(grp.Subs)[subIdx].LastOrderId++;
		et.LfpOrderId = (grp.Subs)[subIdx].LastOrderId;
		
		DbgWriteLog("\nThe id (skill8) of the model from the group %s - %s - Nr. %d was not set.", grp.Name, (grp.Subs)[subIdx].Name, (long)et.LfpOrderId);		
	}
	
	/* Set default values, because queries lead to a crash if there is no value */		
	//et.parent = NULL;
	
	ent_cloneskin(et);
		
	
	
	/* Initial values if they have not been set in WED */
	if(et.LfpLightDist <= 0) { et.LfpLightDist = 500;}
	if(et.LfpRadius <= 0) { et.LfpRadius = 1200;}	
	if(et.LfpMaxDustAlpha <= 0) { et.LfpMaxDustAlpha = 70;}	
	if(et.LfpMaxLightAlpha <= 0) { et.LfpMaxLightAlpha = 70;}
	if(et.LfpMaxAuraAlpha <= 0) { et.LfpMaxAuraAlpha = 50;}
			
	light.Ent = et;
	light.Vertex = vertexNum;
	light.MainGrpId = id;
	light.SubGrpId = sId;
	light.Aura = NULL;
	light.MatAura = NULL;	
	light.IsCreated = false;
	light.IsClipped = light.Ent.LfpVisAlpha;
	light.IsSun = id == LfpLTypePoint && sId == LfpSun1;
	//light.Ent.flags &= ~FLAG8;
	
	if(is(light.Ent, LfpHasDirtyFlares))
	{
		light.Df = malloc(sizeof(LfpFlares));
		light.Df.Count = 125;	
		light.Df.Flares = NULL;
		light.Df.Aura = NULL;
		light.Df.MatAura = NULL;
	}

	light.Sf = malloc(sizeof(LfpFlares));
	light.Sf.Count = 0;		
	light.Sf.Aura = NULL;
	light.Sf.MatAura = NULL;		
	light.Sf.Flares = NULL;
		
	LfpSetLightMaterial(light);
	LfpSetLightParent(light);
	


	Lfp.MaxLights++;

	#ifdef LfpActivateLog
				
		DbgWriteLog("\n\nLfp-Group: %s zur Laufzeit erstellt", grp.Name);		
		DbgWriteLog("\n\nLfp-Subgroup: %s zur Laufzeit erstellt", (grp.Subs)[subIdx].Name);
					
	#endif

}

///
// Add a single light, if group set already to string1 / string2 in WED
//
// <et> light model
// <vertexNum> Vertex on parent model or NULL
void LfpAddLight(ENTITY* et, var vertexNum)
{
	LfpAddLight(et, et.string1, et.string2, vertexNum);
}

///
// Add a single light, if group set already to string1 / string2 in WED
//
// <et> light model
void LfpAddLight(ENTITY* et)
{
	LfpAddLight(et, et.string1, et.string2, NULL);
}



///
// [INIT]
// In the case of entities that have already been read in at level start, 
// these entities can be transferred to this function.
void LfpInitialize(ENTITY** ents, int count)
{
	DbgWriteLog("\n\nLight Flares (Lfp) initialization");	

	int i,j,k;
	ENTITY* et = NULL;	
	int dirtyFlareCount = 125;
	
	Lfp = malloc(sizeof(LfpStruct));
	memset(Lfp,0,sizeof(LfpStruct));
	
	Lfp.SzHalfX = 0.5 * screen_size.x;
	Lfp.SzHalfY = 0.5 * screen_size.y;
	
	/* The dirty flares can be assigned to several lamps.
		To avoid confusion, all flare values must remain the same. */
	Lfp.DfScalesX = malloc(sizeof(var) * dirtyFlareCount);
	Lfp.DfSizesX = malloc(sizeof(var) * dirtyFlareCount);
	Lfp.DfSizesY = malloc(sizeof(var) * dirtyFlareCount);
	
	for(i = 0; i < dirtyFlareCount; i++)
	{
		(Lfp.DfScalesX)[i] = 0.5 + random(1);
		(Lfp.DfSizesX)[i] = random(screen_size.x);
		(Lfp.DfSizesY)[i] = random(screen_size.y);
	}





	if(ents == NULL)
	{
		DbgWriteLog("\nNo models were found whose string1 consecutive letters begin with.");
		return;
	}
	
	DbgWriteLog("\nNumber of light models: %d", (long)count);
	
	/* Group compilation */
	for(i=0; i< count; i++)
	{
		et = ents[i];
		LfpAddLight(et, et.string1, et.string2, NULL);
	}	
	

	#ifdef LfpActivateLog
			
		for(i=0; i < Lfp.GroupCount; i++)
		{
			int subCount = (Lfp.Groups)[i].Count;
			
			DbgWriteLog("\n\nLfp-Group: %s", (Lfp.Groups)[i].Name);
			DbgWriteLog("\n  Lfp-Number of sub-groups: %d", (Lfp.Groups)[i].Count);

			for(j=0; j < subCount; j++)
			{
				LfpSubGroup* subGrp = &(((Lfp.Groups)[i]).Subs)[j];

				DbgWriteLog("\n    Lfp-Subgroup: %s", subGrp.Name);
				DbgWriteLog("\n    Lfp-Number of lights: %d",  subGrp.Count);
			}
		}
	
	#endif
	
	
	/* Cleanup */
	if(ents)
	{
		free(ents);
		ents = NULL;		
	}
}

///
// [INIT]
// This call reads in all models when the game starts and checks the strings string1 and string2.
// If a string is set in string1 with the initials "Lfp" as the first 3 initial letters,
// this will be counted as a group.
// String1 is the main group, string2 is the subgroup. The group assignments are located
// in the header under lighting groups.
void LfpInitialize()
{
	int entCount = 0, grpCount=0;
	ENTITY** ents = NULL;

	/* Find all entities with "Lfp" in String1 as the first letter in the level*/
	
	for(you = ent_next(NULL); you; you = ent_next(you))
	{
		if(you)
		{
			if(you.string1 == NULL || you.string2 == NULL) { continue;}

			if(str_cmpni (you.string1, "Lfp"))
			{
				ents = realloc(ents, sizeof(ENTITY*) * (entCount +1)); 
				ents[entCount] = you;
				entCount++;
			}
		}
	}
	
	LfpInitialize(ents, entCount);
}

///
// [START]
// This call is the entry point and starts a loop, which in turn starts effects / lens flares of the lights
void LfpStartLights()
{
	DbgWriteLog("\nLfp-Lichtsteuerung gestartet");
	
	int i,j,k;
	
	var temp1, temp2;
	var vecDist = 0;
	ENTITY* lf = NULL;
	var vertexPos[3];
	
	while(Lfp)
	{
		/* Main groups - string1 */
		for(i=0; i < Lfp.GroupCount; i++)
		{
			/* Subgroups - string2 */
			int subCount = (Lfp.Groups)[i].Count;
			for(j=0; j < subCount; j++)
			{
				/* Light model with flares */
				LfpSubGroup* subGrp = &(((Lfp.Groups)[i]).Subs)[j];		
				for(k=0; k < subGrp.Count; k++)
				{
					LfpLight* light = &(subGrp.Lights)[k];
					lf = light.Ent;				

					if(light.Vertex)
					{
						vec_for_vertex(vertexPos, lf.parent, light.Vertex);
						vec_set(lf.x, vertexPos);
					}
					
					vecDist = vec_dist(lf.x, camera.x);				
					if(vecDist > lf.LfpLightDist && !light.IsSun)
					{
						lf.LfpVisAlpha = lf.alpha = lf.LfpMaxLightAlpha;
						
						/* If light is outside the specified range, release resources */
						LfpDeleteLightResources(light);		
					}
					else
					{
						/* If light is within the specified range, create flares */
						LfpCreateLight(light);
						
						// With parent entities, the trace can be controlled by C Ignore			
						lf.LfpTracePos = c_trace (lf.x,camera.x,IGNORE_ME|IGNORE_YOU|IGNORE_PASSABLE|USE_POLYGON|IGNORE_SPRITES);				
						if(lf.LfpTracePos <= 0)
						{
							LfpControlLightAuras(light);				
							LfpControlLights(light, vecDist);
										
							/* Position of the lamp, which is converted into view xy coordinates by vec_to_screen */			
							vec_set(lf.LfpPosX, lf.x);
									
							bool isNotVisible = vec_to_screen(lf.LfpPosX, camera) == NULL;
							lf.LfpPosX -= Lfp.SzHalfX;	
							lf.LfpPosY -= Lfp.SzHalfY;				
							
							if(isNotVisible)
							{					
								/* Hide light */
								LfpControlAllLightFlareEffects(light, true);															
							}
							else
							{		
								/* Show light */		
								LfpControlAllLightFlareEffects(light, false);
								LfpControlLightBlending(light, 600);
								
								/* Resolution fading, hides the flares as soon as the screen has been reached */		
								if (light.Df)
								{
									var absX = abs(lf.LfpPosX);
									var absY = abs(lf.LfpPosY);
									var fadeXVal = Lfp.SzHalfX - absX;
									var fadeYVal = Lfp.SzHalfY - absY;
									
									if(absX >= (Lfp.SzHalfX-100)) lf.LfpVisAlpha = fadeXVal;
									if(absY >= (Lfp.SzHalfY-100)) lf.LfpVisAlpha = fadeYVal;
									
									if(absX >= (Lfp.SzHalfX-100) || absY >= (Lfp.SzHalfY-100))
									{
										if(fadeXVal < fadeYVal) {
											lf.LfpVisAlpha = fadeXVal;
										}
										else {
											lf.LfpVisAlpha = fadeYVal;
										}
									}
								}						
							}
	
							if (light.Df) 
							{
								lf.LfpMidPosX = lf.LfpPosX+Lfp.SzHalfX;
								lf.LfpMidPosY = lf.LfpPosY+Lfp.SzHalfY;
							}
						}					
						else
						{
							/* Hide light */
							LfpControlAllLightFlareEffects(light, true);	
						}
					}			
				}
			}
		}
		
		wait(1);	
	}
}

///
// This function cleans up resources if the lights / flares are no longer used.
// The models also clean themselves up if a certain range (defined in Skill1) is exceeded
void LfpCleanUp()
{
	int i,j,k;
	ENTITY* ent = NULL;
	
	for(i=0; i < Lfp.GroupCount; i++)
	{
		int subCount = (Lfp.Groups)[i].Count;
		LfpGroup* grp = &(Lfp.Groups)[i];
		for(j=0; j < subCount; j++)
		{
			LfpSubGroup* subGrp = &(((Lfp.Groups)[i]).Subs)[j];		
			for(k=0; k < subGrp.Count; k++)
			{
				LfpLight* light = &(subGrp.Lights)[k];
				
				ent = light.Ent;	
				
				if(ent.parent)
				{
					ent_purge(ent.parent);
					ptr_remove(ent.parent);
					ent.parent = NULL;
				}
						
				LfpDeleteLightResources(light);
				
				if(ent)
				{
					ent_purge(ent);
					ptr_remove(ent);
					ent = NULL;
					light.Ent = NULL;		
				}
				
				free(light);
				light = NULL;				
			}
			
			ptr_remove(subGrp.Name);
			subGrp.Name = NULL;
			
			if(subGrp.Lights)
			{
				free(subGrp.Lights);
				subGrp.Lights = NULL;			
			}		
			
			free(subGrp);
			subGrp = NULL;
		}
		
		if(grp.Subs)
		{
			free(grp.Subs);
			grp.Subs = NULL;			
		}
		
		ptr_remove(grp.Name);
		grp.Name = NULL;
	}
	
	if(Lfp.Groups)
	{
		free(Lfp.Groups);
		Lfp.Groups = NULL;		
	}

	if(Lfp)
	{
		free(Lfp);
		Lfp = NULL;		
	}

	DbgWriteLog("\n\nCustomLensflares cleanup performed");
}













///
// DUST & DIRTY
// ------------
// Dust Flares are 128 view layers that are spread over the surface of the screen and dirt,
// to show hair or the like on the lens where the light refracts.
//
// Activation: Flag8 (LfpHasDirtyFlares)
///


///
// Controls the alpha values of the dirty lens entities
void LfpCtrlDirtyFlares(LfpLight* light)
{
	if(light.Df == NULL) { return; }
	
	ENTITY* ent = light.Ent;
	int i;
	
	if(light.Df.Aura)
	{
		you = ptr_for_handle(light.Df.Aura);
		if(you)
		{
			you.alpha = ent.skill22;
			you.alpha = clamp(you.alpha, 0, ent.LfpMaxAuraAlpha);
			
			LfpPlace(you, vector(ent.LfpPosX,ent.LfpPosY,ent.LfpPosZ), LfpModeXYP);
			
			/* FX - aura around the center */
			/* Prevent overloading of the light sources */
			you.scale_x = light.Radius/200;
			you.scale_y = light.Radius/200;
			you.flags2 |= SHOW;			
		}			
	}	

	if(light.Df.Flares)
	{
		for(i=0; i < light.Df.Count; i++)
		{
			if((light.Df.Flares)[i])
			{
				you = ptr_for_handle((light.Df.Flares)[i]);
				if(you)
				{
					you.flags2 |= SHOW;
					you.flags |= BRIGHT|TRANSLUCENT;
					you.ambient = 100;
					you.albedo = -100;
					you.x = you.LfpAxisRl;
					you.y = you.LfpAxisUd;
					you.z = 750;
					
					rel_for_screen(you.x,camera);
					
					if (ent.LfpMidPosX < screen_size.x  && ent.LfpMidPosX > 0 && ent.LfpMidPosY < screen_size.y && ent.LfpMidPosY > 0)
					{
						you.LfpTemp1 = vec_dist(you.LfpAxisRl, ent.LfpMidPosX);		
						you.LfpTemp2 = (100/ent.skill5)*you.LfpTemp1;
						you.LfpTemp3 = (ent.skill6/100)*you.LfpTemp2;			
						you.alpha=(ent.LfpMaxDustAlpha-you.LfpTemp3)*(ent.skill22/100)*(ent.LfpVisAlpha/100);  
					}
					else 
					{
						you.alpha = 0; 			
					}
				}		
			}	
		}		
	}
}

///
// Randomly generates an array of view models on the screen.
// This effect always appears when, for example, there is light in one in a darkened room
// certain angle shines through a dirty window.
// The number is set in the LfpInitialize method in the MaxDirtyFlares property.
//
// <light> the light detail struct
var* LfpCreateDirtyFlares(LfpLight* light)
{
	var* dirtyFlares = malloc(sizeof(var) * light.Df.Count);
	int mainId = light.MainGrpId;	
	int subId = light.SubGrpId;			
	ENTITY* ent = light.Ent;		
	int i,rLayer = 10,rImage = 0;
	
	static const char* lfDirtA = "LF_dirtA.tga";
	static const char* lfDirtB = "LF_dirtB.tga";
	static const char* lfDirtC = "LF_dirtC.tga";
	static const char* lfDirtD = "LF_dirtD.tga";	
	static const char* lfDirtE = "LF_dirtE.tga";	
	static const char* lfDirtG = "LF_dirtG.tga";	
	
	for(i=0; i < light.Df.Count; i++)
	{
		if(i == 0) {
			you = ent_createlayer(lfDirtD,0,10);
		}
		else if(i == 1) { 
			you = ent_createlayer(lfDirtG,0,10);
		}
		else
		{
			rLayer = 7 + integer(random(11));
			rImage = integer(random(5));
			
			switch(rImage)
			{
				case 0: you = ent_createlayer(lfDirtA,0,rLayer); break;
				case 1: you = ent_createlayer(lfDirtB,0,rLayer); break;
				case 2: you = ent_createlayer(lfDirtC,0,rLayer); break;
				case 3: you = ent_createlayer(lfDirtE,0,rLayer); break;
				default: you = ent_createlayer(lfDirtA,0,rLayer); break;
			}
		}

		#ifdef SSAO
			you.client_id = viewSsaoCombine;
		#endif

		you.scale_x = (Lfp.DfScalesX)[i];
		you.scale_y = you.scale_x;
		you.LfpAxisRl = (Lfp.DfSizesX)[i];
		you.LfpAxisUd = (Lfp.DfSizesY)[i];
		you.roll = 90;
		
		
		dirtyFlares[i] = handle(you);
	}

	//	DbgWriteLog("\nNumber of DirtyFlares: %d", (long)light.DirtyFlareCount);
	DbgWriteLog("\nDirtyFlares and aura created by %s-%s.", LfpGetGrpText(mainId, false), LfpGetGrpText(subId, true));
	
	return dirtyFlares;
}

///
// Removes all dirty flare resources, including the aura.
//
// <light> the light detail struct
void LfpDeleteDirtyFlares(LfpLight* light)
{
	if(light.Df == NULL) { return; }

	int i;
	
	if(light.Df.Flares)
	{
		//DbgWriteLog("\nDelete dirty flares: %s-%s-%d", _chr(light.Ent.string1),  _chr(light.Ent.string2), (long)light.Ent.LfpOrderId);
		for(i=0; i < light.Df.Count; i++)
		{
			you = ptr_for_handle((light.Df.Flares)[i]);		
			if(you)
			{
				ent_purge(you);			
				ptr_remove(you);
				you = NULL;
			}
		}
	
		free(light.Df.Flares);
		light.Df.Flares = NULL;
	}

	if(light.Df.Aura)
	{
		you = ptr_for_handle(light.Df.Aura);
		if(you)
		{
			if(light.Df.MatAura)
			{
				you.material = NULL;
				ptr_remove(light.Df.MatAura);
				light.Df.MatAura = NULL;
				//DbgWriteLog("\nDust-aura material of the group: %s-%s-Nr. %d. gelscht", _chr(light.Ent.string1),  _chr(light.Ent.string2), (long)light.Ent.LfpOrderId);		
			}
			
			ent_purge(you);
			ptr_remove(you);
			you = NULL;
			light.Df.Aura = NULL;
		}
	}
}













///
// Flare-Controling
// ------------
// 
/// 


void LfpCtrlLightsGrpPointAdd1(ENTITY* ent, ENTITY* entFlare, int index, var val1, var val2)
{
	if (ent.skill21 > 0) 
	{
		entFlare.alpha=(((val1)+random(val2)) * ent.LfpVisAlpha) * (ent.skill21/100);
	}
	else 
	{
		entFlare.alpha = 0;
	}	  
}

void LfpCtrlLightsGrpPointAdd2(ENTITY* ent, ENTITY* entFlare, int index, var val1, var val2)
{
	entFlare.alpha = val1 * ent.LfpVisAlpha + random((val2) * ent.LfpVisAlpha);	
}



/// 
// The functions place view layers (lens flares) at specific coordinates on the screen
//
// <ent> view entity
// <pos> Position of the entity in space
// <mode> LfpModeD 0 | LfpModeX 1 | LfpModeXY 2 | LfpModeXYP 3
void LfpPlace(ENTITY* ent, var* pos, var mode)
{
	vec_set(ent.LfpAxisRl, pos); 
	
	ent.flags2 |= SHOW;
	ent.flags |= BRIGHT|TRANSLUCENT|NOFOG;	
	ent.ambient = 100;
	ent.albedo = -100;
	
	switch(mode)
	{
		// Places the view ents in X mode (additional X movement)	
		case LfpModeX:
			ent.x = ent.LfpAxisRl * ent.LfpPivotDist + Lfp.SzHalfX + (ent.LfpAxisRl * ent.LfpAxisPdRl);	
			ent.y = ent.LfpAxisUd * ent.LfpPivotDist + Lfp.SzHalfY;	
			break;
		// Places the view ents in XY mode (additional XY movement)
		case LfpModeXY:		
			ent.x = ent.LfpAxisRl * ent.LfpPivotDist + Lfp.SzHalfX + (ent.LfpAxisRl * ent.LfpAxisPdRl);	
			ent.y = ent.LfpAxisUd * ent.LfpPivotDist + Lfp.SzHalfY + (ent.LfpAxisUd * ent.LfpAxisPdUd);
			break;
		// Places the view ents in the middle + Pivot Dist
		case LfpModeXYP:
			ent.x = ent.LfpAxisRl * ent.LfpPivotDist + Lfp.SzHalfX;	
			ent.y = ent.LfpAxisUd * ent.LfpPivotDist + Lfp.SzHalfY;			
			break;
		default:
			ent.x = ent.LfpAxisRl;	
			ent.y = ent.LfpAxisUd;
			break;	
	}
	
	ent.z = 750;
	
	rel_for_screen(ent.x,camera);
}

///
// Sets the color (s) of the specified lens flares.
//
// <src> Array of handles, with which the lens flares can be called up
// <light> The light model
// <colors> color array for the lens flares
void LfpSetLightFlareColor(var* src, LfpLight* light, var* colors)
{
	int i,ig=0;

	for(i=0; i < light.Sf.Count; i++)
	{
		you = ptr_for_handle(src[i]);
		if(you)
		{
			ig = i * 3;
			
			if(colors[ig] > -1) 	 you.blue = colors[ig];
			if(colors[ig+1] > -1) you.green = colors[ig+1];
			if(colors[ig+2] > -1) you.red = colors[ig+2];
		}
	}
}

///
// Creates a lens flare view with the given name and returns it as a handle
//
// <name> Name of the lens flare file
// <layer> view layer
// <pd> Pivot Dist (start position of the view)
// <scale> Array (Scale x, y, z) for the scaling
// <axisPdRl> View position on the horizontal axis
// <axisPdUd> View position on the vertical axis
var LfpCreateLightFlare(char* name, var layer, var pd, var* scale, var axisPdRl, var axisPdUd)
{
	you = ent_createlayer(name,0,layer);
	
	#ifdef SSAO
		you.client_id = viewSsaoCombine;
	#endif
		
	you.LfpPivotDist = pd;
	you.alpha = 0;
	vec_set(you.scale_x, scale);
	you.LfpAxisPdRl = axisPdRl; 	
	you.LfpAxisPdUd = axisPdUd;

	return handle(you);
}

///
// Deletes the flares and aura of the given light model
//
// <ent> the light model
void LfpDeleteLightFlares(LfpLight* light)
{
	if(light.Sf == NULL) { return; }
	
	int i;
	
	if(light.Sf.Flares)
	{
		for(i=0; i < light.Sf.Count; i++)
		{
			you = ptr_for_handle((light.Sf.Flares)[i]);	
			if(you)
			{
				ent_purge(you);
				ptr_remove(you);
				you = NULL;
			}
		}
		
		free(light.Sf.Flares);
		light.Sf.Flares = NULL;
	}
	
	if(light.Sf.Aura)
	{
		you = ptr_for_handle(light.Sf.Aura);
		if(you)
		{
			if(light.Sf.MatAura)
			{
				you.material = NULL;
				ptr_remove(light.Sf.MatAura);
				light.Sf.MatAura = NULL;
				//DbgWriteLog("\nMaterial of the group: %s-%s-Nr. %d. deleted", _chr(light.Ent.string1),  _chr(light.Ent.string2), (long)light.Ent.LfpOrderId);		
			}

			ent_purge(you);
			ptr_remove(you);
			light.Sf.Aura = NULL;
		}
	}
}

///
// Returns an array of lens flares as view entities.
// This function defines the colors and lens flare types.
//
// <light> the light detail struct
// <ent> the light model
var* LfpCreateLightFlares(LfpLight* light)
{
	var* ret = NULL;	
	int mainId = light.MainGrpId;	
	int subId = light.SubGrpId;
	ENTITY* ent = light.Ent;

	static const char* lfSunA = "LF_sunA.tga";
	static const char* lfSunB = "LF_sunB.tga";	
	static const char* lfSunC = "LF_sunC.tga";
	static const char* lfSunD = "LF_sunD.tga";
	static const char* lfSunE = "LF_sunE.tga";
	static const char* lfSunF = "LF_sunF.tga";
	static const char* lfSunG = "LF_sunG.tga";		
	static const char* lfSunH = "LF_sunH.tga";
	static const char* lfSunI = "LF_sunI.tga";	
	static const char* lf10Grp9 = "LF_10grp9.tga";	
	static const char* lf10Grp4 = "LF_10grp4.tga";
	static const char* lf10Grp10 = "LF_10grp10.tga";
	static const char* lfEntSpez = "lf_ent_spz.mdl";	
	
		
	if(mainId == LfpLTypeLong)
	{
		/* NEONRHRE 1 */
		if(subId == LfpNeonPipe1)
		{
			light.Sf.Count = 10;
			ret = malloc(sizeof(var) * light.Sf.Count);
			
			ret[0] = LfpCreateLightFlare(lf10Grp9, 5, 1,	vector(1,1,0),			0,0);			
			ret[1] = LfpCreateLightFlare(lf10Grp4, 2, 1,	vector(1,8,0), 		-0.5,0);
			ret[2] = LfpCreateLightFlare(lf10Grp9, 1, 1.2,	vector(1,1,0),			-2,0.4);
			ret[3] = LfpCreateLightFlare(lf10Grp9, 1, 1.2,	vector(1,1,0),			-1,0.4);		
			ret[4] = LfpCreateLightFlare(lf10Grp10,6, 1.2,	vector(1,1,0),			-1,-1);
			ret[5] = LfpCreateLightFlare(lf10Grp9, 1, 1,	vector(1,1,0),			0.8,-0.1);
			ret[6] = LfpCreateLightFlare(lfEntSpez,1, 1,	vector(0,300,300),	0,-0.3);
			ret[7] = LfpCreateLightFlare(lf10Grp9, 1, -0.7,vector(1.8,	0.4,0),	0,0);
			ret[8] = LfpCreateLightFlare(lf10Grp9, 1, -0.3,vector(1.5,	0.2,0),	0,0);	
			ret[9] = LfpCreateLightFlare(lf10Grp9, 1, -1,	vector(2.2,	0.5,0),	0,0);
			
			/*
				Define colors (of the flares)
				3 values result in an index. All values (number of flareCount * 3) must be specified.
				If certain indices are not to be colored, the values must be specified with -1.
			*/			
			var col1[30] = { 0,0,0, 10,10,220, 100,250,120, 90,100,200,  -1,-1,-1, 250,50,190, 10,10,120, 30,10,70, 20,100,10, 30,80,20};				
			var col2[30] = { 0,0,0, 200,50,0,  220,120,100, 200,80,170, -1,-1,-1, 200,250,30, 80,130,50, 80,10,30, 10,60,20,  80,20,80};			
			var col3[30] = { 0,0,0, 50,200,50, 150,190,10,	 100,200,70, -1,-1,-1, 0,160,190,  20,120,50, 20,80,10, 10,60,20,  80,20,120};
			var col4[30] = { 0,0,0, 180,0,50,  100,220,100, 200,100,130, -1,-1,-1, 200,30,170, 120,50,80, 70,20,10, 70,20,10,  70,20,10 }; 
			
			/* Arrays can only be initialized with numbers. Skills and other must be added afterwards */	
			col1[0]=col2[0]=col3[0]=col4[0] = ent.skill2;			
			col1[1]=col2[1]=col3[1]=col4[1] = ent.skill3;
			col1[2]=col2[2]=col3[2]=col4[2] = ent.skill4;
			
			/* A lighting group can have several subgroups with different models (skill8) */
			if(ent.LfpOrderId == 1) { LfpSetLightFlareColor(ret,light,col1);}
			if(ent.LfpOrderId == 2) { LfpSetLightFlareColor(ret,light,col2);}
			if(ent.LfpOrderId == 3) { LfpSetLightFlareColor(ret,light,col3);}
			if(ent.LfpOrderId == 4) { LfpSetLightFlareColor(ret,light,col4);}	
		}
		/* NEONRHRE 2 */
		else if(subId == LfpNeonPipe2)
		{
			light.Sf.Count = 5;
			ret = malloc(sizeof(var) * light.Sf.Count);
			
			/* Define Flares */		
			ret[0] = LfpCreateLightFlare(lf10Grp9, 3, 1,	vector(1,1,0),		-1,0.4);
			ret[1] = LfpCreateLightFlare(lf10Grp9, 1, 1.2,	vector(1,1,0),		0,0);
			ret[2] = LfpCreateLightFlare(lf10Grp9, 1, -0.3,vector(1.8,0.4,0),0,0);
			ret[3] = LfpCreateLightFlare(lf10Grp9, 1, -0.7,vector(2,0.6,0),	0,0);
			ret[4] = LfpCreateLightFlare(lf10Grp10, 1, 1.2,vector(1,1,0),		-1,-1);

			/* Define colors (of the flares) */
			var col1[15] = { 100,100,250, 250,100,180,  70,70,180,   150,150,250,  100,100,250}; 
			var col2[15] = { 250,100,100, 200,250,100,  180,120,80,  180,120,80,   100,100,250};					
			var col3[15] = { 70,250,70,   70,250,180,   10,100,250,  100,200,250,  100,100,250};				
			var col4[15] = { 210,220,100, 250,120,100,  100,120,140, 100,120,140,  100,100,250};

			if(ent.LfpOrderId == 1) { LfpSetLightFlareColor(ret,light,col1); }
			if(ent.LfpOrderId == 2) { LfpSetLightFlareColor(ret,light,col2); }
			if(ent.LfpOrderId == 3) { LfpSetLightFlareColor(ret,light,col3); }
			if(ent.LfpOrderId == 4) { LfpSetLightFlareColor(ret,light,col4); }
		}	
	}
	else if(mainId == LfpLTypePoint)
	{
		/* SONNE 1 */	
		if(subId == LfpSun1)
		{
			light.Sf.Count = 23;
			ret = malloc(sizeof(var) * light.Sf.Count);
			
			ret[0] = LfpCreateLightFlare(lfSunA, 1, 	1, 	vector(2,2,0), 	0,0);
			ret[1] = LfpCreateLightFlare(lfSunA, 1, 	1, 	vector(2.3,2.3,0),0,0);
			ret[2] = LfpCreateLightFlare(lfSunC, 10, 1, 		vector(13,13,0),	0,0);
			ret[3] = LfpCreateLightFlare(lfSunD, 2, 	0.96, vector(1,1,0),		0,0.1);
			ret[4] = LfpCreateLightFlare(lfSunE, 2, 	0.92, vector(1,1,0),		0,0);
			ret[5] = LfpCreateLightFlare(lfSunF, 3, 	0.92, vector(1,1,0),		0,0);
			ret[6] = LfpCreateLightFlare(lfSunF, 3, 	0.92, vector(1,1,0),		0,0);
			ret[7] = LfpCreateLightFlare(lfSunF, 3, 	0.89, vector(2,0.7,0),	0,0);
			ret[8] = LfpCreateLightFlare(lfSunF, 3, 	0.89, vector(2,0.7,0),	0,0);
			ret[9] = LfpCreateLightFlare(lfSunG, 1, 	1, 	vector(1,1,0),		0,0);
			ret[10] = LfpCreateLightFlare(lfSunH, 1, 	1.08, vector(1,1,0),		0,0);
			ret[11] = LfpCreateLightFlare(lfSunI, 1, 	1.2, 	vector(1.2,1.2,0),0,0);
			ret[12] = LfpCreateLightFlare(lfSunI, 1, 	1.35, vector(3.6,3.6,0),0,0);
			ret[13] = LfpCreateLightFlare(lfSunI, 1, 	0.76, vector(0.6,0.6,0),0,0);
			ret[14] = LfpCreateLightFlare(lfSunI, 1, 	0.55, vector(0.2,0.2,0),0,0);
			ret[15] = LfpCreateLightFlare(lfSunI, 1, 	0.33, vector(0.8,0.8,0),0,0);
			ret[16] = LfpCreateLightFlare(lfSunI, 1, 	0.2, 	vector(0.1,0.1,0),0,0);
			ret[17] = LfpCreateLightFlare(lfSunI, 1, 	0.05, vector(2,2,0),		0,0);
			ret[18] = LfpCreateLightFlare(lfSunI, 2, 	0.01, vector(1,1,0),		0,0);
			ret[19] = LfpCreateLightFlare(lfSunI, 1, 	-0.5, vector(2,2,0),		0,0);
			ret[20] = LfpCreateLightFlare(lfSunI, 3, 	-1, 	vector(7,7,0),		0,0);			
			/* Special effects can be created with models */		
			ret[21] = LfpCreateLightFlare(lfEntSpez, 4, -1.35, vector(0,25,25),0,0);
			ret[22] = LfpCreateLightFlare(lfEntSpez, 3, -0.85, vector(0,3,3),	0,0);
			
			var col1[63] = { -1,-1,-1, -1,-1,-1, -1,-1,-1, -1,-1,-1, -1,-1,-1, 0,0,0, 0,0,0, 0,0,0, 0,0,0, -1,-1,-1, 
				-1,-1,-1, 10,200,220,  50,220,250,  0,180,25,	100,20,80, 200,35,140, 80,5,50, 120,60,10, 200,20,40, 200,20,40, 
			180,250,45 };

			if(ent.LfpOrderId == 1) { LfpSetLightFlareColor(ret,light,col1); }	
		}
		/* SONNE 2 */			
		else if(subId == LfpSun2)	
		{
			light.Sf.Count = 26;
			
			ret = malloc(sizeof(var) * light.Sf.Count);		
			
			ret[0] = LfpCreateLightFlare(lfSunA, 	2, 1,		vector(4,4,0),			-1.1,-1.1);
			ret[1] = LfpCreateLightFlare(lfSunA, 	1, 1,		vector(4.2,4.2,0), 	0,0);
			ret[2] = LfpCreateLightFlare(lfSunB, 	3, 1,		vector(10,10,0), 		0,0);
			ret[3] = LfpCreateLightFlare(lfSunH, 	5, 1.08,	vector(1,1,0), 		-1.1,-1.1);
			ret[4] = LfpCreateLightFlare(lfSunD, 	2, 0.96,	vector(1,1,0), 		0,0);
			ret[5] = LfpCreateLightFlare(lfSunI, 	1, 0.76,	vector(0.8,0.8,0), 	0,0);
			ret[6] = LfpCreateLightFlare(lfSunI, 	1, 0.55,	vector(1.2,1.2,0), 	0,0);
			ret[7] = LfpCreateLightFlare(lfSunI, 	1, 0.58,	vector(0.7,0.7,0), 	0,0);	
			ret[8] = LfpCreateLightFlare(lfSunB, 	1, 0.53,	vector(0.6,0.6,0), 	0,0);
			ret[9] = LfpCreateLightFlare(lfSunI, 	1, 0.6,	vector(1.6,1.6,0), 	0,0);
			ret[10] = LfpCreateLightFlare(lfSunI, 	1, 0.38,	vector(1.2,1.2,0), 	0,0);
			ret[11] = LfpCreateLightFlare(lfSunI, 	1, 0.3,	vector(0.6,0.6,0), 	0,0);
			ret[12] = LfpCreateLightFlare(lfSunB, 	1, 0.29,	vector(3,3,0), 		0,0);
			ret[13] = LfpCreateLightFlare(lfSunB, 	1, 0.2,	vector(0.3,0.3,0), 	0,0);
			ret[14] = LfpCreateLightFlare(lfSunB, 	1, 0.1,	vector(0.6,0.6,0), 	0,0);
			ret[15] = LfpCreateLightFlare(lfSunB, 	1, 0.01,	vector(5,0.4,0), 		0,0);
			ret[16] = LfpCreateLightFlare(lfSunI, 	1, -0.05,vector(1.6,1.6,0), 	0,0);
			ret[17] = LfpCreateLightFlare(lfSunI, 	1, -0.1,	vector(0.6,0.6,0), 	0,0);
			ret[18] = LfpCreateLightFlare(lfSunI, 	1, -0.35,vector(0.9,0.9,0), 	0,0);
			ret[19] = LfpCreateLightFlare(lfSunI, 	1, -0.3,	vector(1.7,1.7,0), 	0,0);
			ret[20] = LfpCreateLightFlare(lfSunI, 	1, -0.48,vector(2,2,0), 		0,0);	
			ret[21] = LfpCreateLightFlare(lfSunI, 	1, -0.65,vector(0.8,0.8,0), 	0,0);
			ret[22] = LfpCreateLightFlare(lfSunI, 	1, -1,	vector(2.7,2.7,0), 	0,0);
			ret[23] = LfpCreateLightFlare(lfSunI, 	1, -2,	vector(1.7,1.7,0), 	0,0);
			ret[24] = LfpCreateLightFlare(lfSunI, 	1, -2,	vector(13,13,0), 		0,0);
			ret[25] = LfpCreateLightFlare(lfEntSpez, 1, -1.5,	vector(45,45,0), 	0,0);
			
			var col1[75] = { -1,-1,-1,  -1,-1,-1,  -1,-1,-1,  -1,-1,-1,  -1,-1,-1,  100,220,240,  100,180,230,	10,120,180,  200,100,100,  200,150,250, 
				250,50,200, 190,0,90,  200,150,10,  100,30,0,  200,100,50, 200,80,150, 190,220,30,  100,100,230,  0,0,180, 200,100,180, 
			200,20,150, 150,50,0,  250,150,110, 200,250,100, 150,50,0 }; 														
			
			if(ent.LfpOrderId == 1) { LfpSetLightFlareColor(ret,light,col1); }					
		}
		/* STEHLAMPE */		
		else if(subId == LfpLamp1)	
		{
			light.Sf.Count = 10;
			ret = malloc(sizeof(var) * light.Sf.Count);
			
			/* 1.flare is a star-shaped flare (LF_sunA.tga). Type1 refers to lantern light flares.
				This slowly revolves around its own axis (roll value!)*/
			ret[0] = LfpCreateLightFlare(lfSunA, 5, 1,	vector(2,2,0),		0,0);			
			/* 2. flare is also a star-shaped flare, but turns in the opposite direction as A. TYPE1. */
			ret[1] = LfpCreateLightFlare(lfSunA, 4, 1,	vector(2.1,2.1,0),0,0);			
			/* 3. flare of the lantern light, 2 outer crescents */
			ret[2] = LfpCreateLightFlare(lfSunG, 3, 1,	vector(2,2,0),		0,0);				
			/* 4. flare of the lantern light, long blue line that widens when the camera goes out of view	 */
			ret[3] = LfpCreateLightFlare(lfSunE, 2, 1,	vector(1,1,0),		0,0);					
			/* 5. flare of the lantern light, 7-sided pattern that appears above the light source	 */
			ret[4] = LfpCreateLightFlare(lfSunI, 4, 1.2,vector(2,2,0),		0,0);					
			/* 6. flare of lantern light, round pattern */
			ret[5] = LfpCreateLightFlare(lfSunB, 2, 0.8,vector(0.6,0.6,0),	0.5,0);					
			/* 7,8,9. flare of lantern light, 7-sided pattern, below the light source	 */
			ret[6] = LfpCreateLightFlare(lfSunI, 2, 0.5,vector(0.6,0.6,0),	-0.5,0);			
			ret[7] = LfpCreateLightFlare(lfSunI, 3, -0.05,vector(1.2,1.2,0), 1.1,0);
			ret[8] = LfpCreateLightFlare(lfSunI, 3, -0.9,vector(2.8,2.8,0),-1.1,0);		
			/* 10. flare of the lantern light, special flare (model entity), below the light source, which can also be animated */
			ret[9] = LfpCreateLightFlare(lfEntSpez, 1, -1.8,vector(0,40,40),0,0);

			var col1[30] = { 0,0,0, 0,0,0, 0,0,0, -1,-1,-1, -1,-1,-1, 80,80,200 ,  180,80,200, 0,130,100,  150,50,70 	,-1,-1,-1}; 
			var col2[30] = { 0,0,0, 0,0,0, 0,0,0, -1,-1,-1, -1,-1,-1, 80,160,180,  180,100,80, 180,200,0,   100,180,200 ,-1,-1,-1};
			var col3[30] = { 0,0,0, 0,0,0, 0,0,0, -1,-1,-1, -1,-1,-1, 50,170,50,   50,170,180, 250,70,200,  130,180,120	,-1,-1,-1};
			var col4[30] = { 0,0,0, 0,0,0, 0,0,0, -1,-1,-1, -1,-1,-1, 180,150,200, 50,170,50,  200,100,250, 160,100,190	,-1,-1,-1};	
			
			col1[0]=col2[0]=col3[0]=col4[0] = ent.skill2;			
			col1[1]=col2[1]=col3[1]=col4[1] = ent.skill3;
			col1[2]=col2[2]=col3[2]=col4[2] = ent.skill4;	
			col1[3]=col2[3]=col3[3]=col4[3] = ent.skill2;					
			col1[4]=col2[4]=col3[4]=col4[4] = ent.skill3;
			col1[5]=col2[5]=col3[5]=col4[5] = ent.skill4;		 			

			if(ent.LfpOrderId == 1) { LfpSetLightFlareColor(ret,light,col1); }
			if(ent.LfpOrderId == 2) { LfpSetLightFlareColor(ret,light,col2); }
			if(ent.LfpOrderId == 3) { LfpSetLightFlareColor(ret,light,col3); }
			if(ent.LfpOrderId == 4) { LfpSetLightFlareColor(ret,light,col4); }
		}
	}
	else if(mainId == LfpLTypeWindow)
	{
		
	}
	else
	{
		
	}
	
	DbgWriteLog("\nLensflares of the lighting group \"%s-%s-Nr. %d\" initialized", LfpGetGrpText(mainId, false), LfpGetGrpText(subId,true), (long)ent.LfpOrderId);

	return ret;	
}



///
// This call recalculates the positions of all lens flares for the specified light model every frame
//
// <light> the light detail struct
// <ent> the light model
void LfpPlaceLightFlare(ENTITY* ent, ENTITY* entFlare, int mainId, int subId, int index)
{
	var* vec = vector(ent.LfpPosX,ent.LfpPosY,ent.LfpPosZ);
	
	if(mainId == LfpLTypeLong)
	{
		/* NEONRHRE 1 */
		if(subId == LfpNeonPipe1)
		{
			switch(index)
			{
				case 1: LfpPlace(entFlare, vec,LfpModeX); break;
				case 2:
				case 3:
				case 4:
				case 5:
				case 6:	LfpPlace(entFlare, vec,LfpModeXY); break;
				default: LfpPlace(entFlare, vec,LfpModeXYP); break;
			}
		}
		/* NEONRHRE 2 */
		else if(subId == LfpNeonPipe2)
		{
			switch(index)
			{
				case 2:
				case 4: LfpPlace(entFlare, vec,LfpModeXY); break;			
				default:LfpPlace(entFlare, vec,LfpModeXYP); break;
			}
		}			
	}
	else if(mainId == LfpLTypePoint)
	{	
		/* SONNE 1 */		
		if(subId == LfpSun1)		
		{
			switch(index)
			{
				case 5:
				case 6:
				case 7:
				case 8: LfpPlace(entFlare, vec, LfpModeX); break;			
				case 10:LfpPlace(entFlare, vec, LfpModeXY); break;
				default:LfpPlace(entFlare, vec, LfpModeXYP); break;
			}
		}
		/* SONNE 2 */			
		else if(subId == LfpSun2)
		{
			switch(index)
			{
				case 3: LfpPlace(entFlare, vec,LfpModeXY); break;				
				default:LfpPlace(entFlare, vec,LfpModeXYP); break;									
			}			
		}
		/* STEHLAMPE */		
		else if(subId == LfpLamp1)	
		{
			switch(index)
			{
				case 3: LfpPlace(entFlare, vec,LfpModeXY); break;
				default:LfpPlace(entFlare, vec,LfpModeXYP); break;								
			}		
		}
	}
	else if(mainId == LfpLTypeWindow)
	{
		
	}
	else
	{
	}
}

///
// This call calculates special effects of individual lens flares.
//
// <light> the light detail struct

void LfpControlLightFlare(ENTITY* ent, ENTITY* entFlare,int mainId, int subId, int index)
{
	if(mainId == LfpLTypeLong)
	{
		/* NEONRHRE 1 */
		if(subId == LfpNeonPipe1)
		{
			switch(index)
			{
				case 0:
					entFlare.scale_x = 5 * (ent.skill21/100);
					entFlare.scale_y = 10 * (ent.skill21/100);
					break;		
				case 1:
					entFlare.scale_x = 0.5 + abs(ent.LfpPosX/5);
					break;
				case 2:
					entFlare.scale_y = 0.2 + (abs(ent.LfpPosX/Lfp.SzHalfX)*(ent.skill21/30));
					entFlare.scale_x=entFlare.scale_y+0.7;
					break;		
				case 3:
					entFlare.scale_y = 0.3 + (abs(ent.LfpPosX/200)*(ent.skill21/100));
					entFlare.scale_x=entFlare.scale_y;   	
					break;	
				case 4: 		   	
					entFlare.scale_x = ent.skill50;
					entFlare.scale_x = maxv (3, entFlare.scale_x);
					entFlare.scale_y = entFlare.scale_x;				
					entFlare.roll = ent.skill55-(ent.skill55*2)-45;		   	
					break;		
				case 5:
					entFlare.scale_y = 0.1 + (abs(ent.LfpPosX/Lfp.SzHalfX)*(ent.skill21/25));	
					entFlare.scale_x = 0.5 + abs(ent.LfpPosX/150);
					break;	
				case 6:
					if (mouse_force.x > 0 || mouse_force.x < 0) {
						entFlare.v += (camera.pan*camera.pan);
					} 
					else {
						entFlare.v += 0.4*time_step;
					}
					
					if (mouse_force.y > 0 || mouse_force.y < 0) {
						entFlare.u += (camera.tilt*camera.tilt);
					} 
					else {
						entFlare.u -= 0.1*time_step;
					}		   	
					break;					   				   				   											   
			}
		}
		/* NEONRHRE 2 */
		else if(subId == LfpNeonPipe2)
		{
			switch(index)
			{
				case 0:
					entFlare.scale_x = 5 * (ent.skill21/100);
					entFlare.scale_y = 4 * (ent.skill21/100);
					break;
				case 1:
					entFlare.scale_x = 2 * (ent.skill21/100);
					
					if(ent.skill31 != 0)
					{
						entFlare.scale_y = abs(0.8 * (ent.skill21/ent.skill31));
					}
					else
					{
						entFlare.scale_y = abs(0.8 * (ent.skill21));
					}
					break;
				case 4:
					entFlare.scale_x = ent.skill50;
					entFlare.scale_x = maxv(3, entFlare.scale_x);
					entFlare.scale_y = entFlare.scale_x;								
					entFlare.roll = ent.skill55-(ent.skill55*2)-45;
					break;								
				
			}			
		}
	}
	else if(mainId == LfpLTypePoint)
	{
		/* SONNE 1 */ 
		if(subId == LfpSun1)
		{
			switch(index)
			{
				case 0: entFlare.roll += 0.03 * time_step; break;
				case 1: entFlare.roll -= 0.06 * time_step; break;
				
				case 3:
					entFlare.scale_x = abs(ent.LfpPosX/60);
					entFlare.scale_y = abs(ent.LfpPosX/100);
					break;
					
				case 4:
					entFlare.scale_x = abs(ent.LfpPosX/200);
					entFlare.scale_y = abs(ent.LfpPosX/100);						
					break;
					
				case 5:
				case 6:
					entFlare.scale_x = abs(ent.LfpPosX/700);
					entFlare.scale_y = abs(ent.LfpPosX/350);					
					break;
				
				case 7:
				case 8:	
					entFlare.scale_x = abs(ent.LfpPosX/100);
					break;
					
				case 9:
					// Rainbow Flare: Located at the center and stretching in the opposite direction to the camera.

					var t1[3];
					t1[0] = abs(ent.LfpPosX) - (abs(ent.LfpPosX)-abs(ent.LfpPosX));
					t1[1] = abs(ent.LfpPosY) - (abs(ent.LfpPosY)-abs(ent.LfpPosY));
					
					ent.skill95 = sqrt(t1[0]*t1[0] + t1[1]*t1[1]); 
					
					entFlare.scale_x = ent.skill95/100;
					entFlare.scale_y = ent.skill95/100;		
								
					// additional rotation
					vec_diff(ent.skill30, camera.pan, camera.tilt);
					entFlare.roll = ent.skill30-60; 
					break;
				
				case 10:
					// Special ring flare (rainbow)
				
					entFlare.scale_x = ent.skill95/250;
					entFlare.scale_y = ent.skill95/250;
					
					var t2[3];	
					t2[0] = (ent.LfpPosX - ent.LfpPosX);
					t2[1] = (ent.LfpPosY - ent.LfpPosY);
							
					vec_set(ent.skill50, ent.LfpPosX);
					vec_sub(ent.skill50, t2[0]); 
					vec_to_angle(ent.skill55, ent.skill50);					
					entFlare.roll = ent.skill55-(ent.skill55*2)-90;
					break;
				
				case 21:	
				case 22:				
					/* 
						A few additional effects for these special flares
						Move the texture after camera movement.
						These two flares are view-ents.
					*/
					if (mouse_force.x > 0 || mouse_force.x < 0) {
						entFlare.v += (camera.pan*camera.pan);
					} 
					else {
						entFlare.v += 0.4*time_step;				
					}
					
					if (mouse_force.y > 0 || mouse_force.y < 0) {
						entFlare.u += (camera.tilt*camera.tilt);
					} 		
					else {
						entFlare.u -= 0.1*time_step;
					}			
											
					entFlare.roll = camera.pan*10;
					break;
			}				
		}
		/* SONNE 2 */
		else if(subId == LfpSun2)
		{
			switch(index)
			{	
				case 0:entFlare.roll += 0.03 * time_step; break;	
				case 1:entFlare.roll -= 0.06 * time_step; break;			
				case 3:
					entFlare.scale_x = ent.skill95/250;
					entFlare.scale_y = ent.skill95/250;		
					entFlare.roll = ent.skill55-(ent.skill55*2)-90;
					break;
				case 4:
					entFlare.scale_x = abs(ent.LfpPosX/70);
					entFlare.scale_y = abs(ent.LfpPosX/100);
					break;
				case 15:
					entFlare.roll = ent.skill55-(ent.skill55*2)-90;		
					entFlare.scale_x = abs(ent.LfpPosX/180);
					break;	
				case 25:
					if (mouse_force.x > 0 || mouse_force.x < 0) {
						entFlare.v += (camera.pan*camera.pan);
					} 	
					else {
						entFlare.v += 0.4 *time_step;
					}
					
					if (mouse_force.y > 0 || mouse_force.y < 0) {
						entFlare.u += (camera.tilt*camera.tilt);
					} 	
					else {
						entFlare.u -= 0.1*time_step;
					}
					break;					
			}							
		}
		/* STEHLAMPE */
		else if(subId == LfpLamp1)
		{
			switch(index)
			{
				case 0: entFlare.roll += 0.03 * time_step;break;	
				case 1: entFlare.roll -= 0.07 * time_step;break;			
				case 3:
					entFlare.scale_x = abs(ent.LfpPosX/100);
					entFlare.scale_y = abs(ent.LfpPosX/50);
					break;			
				case 9:
					if (mouse_force.x > 0 || mouse_force.x < 0) {
						entFlare.v += (camera.pan*camera.pan);
					} 	
					else {
						entFlare.v += 0.4 * time_step;
					}
					
					if (mouse_force.y > 0 || mouse_force.y < 0) {
						entFlare.u += (camera.tilt*camera.tilt);
					} 	
					else {
						entFlare.u -= 0.1 * time_step;
					}
					break;	
			}	
		}
	}
	else if(mainId == LfpLTypeWindow)
	{
		
	}
	else
	{
		
	}
}

///
// This call recalculates the alpha values of the individual lens flares for the specified light model every frame
//
// <light> the light detail struct
void LfpAlphaLightFlare(ENTITY* ent, ENTITY* entFlare,int mainId, int subId, int index)
{
	entFlare.flags2 |= SHOW;
	
	if(mainId == LfpLTypeLong)
	{
		/* NEONRHRE 1 */
		if(subId == LfpNeonPipe1)
		{
			switch(index)
			{
				case 0:entFlare.alpha= ((ent.LfpMaxLightAlpha/100) * (ent.skill21/100) * ent.LfpVisAlpha) -15;	break;  	   
				case 1:entFlare.alpha= 1.5	* (ent.skill21/100) * ent.LfpVisAlpha; break;
				case 2:entFlare.alpha= 0.25  * (ent.skill21/100) * ent.LfpVisAlpha; break;
				case 3:entFlare.alpha= 0.6	* (ent.skill21/100) * ent.LfpVisAlpha; break;
				case 4:
				
					if (ent.skill32 > 0) 
					{
						entFlare.LfpTemp1 = ent.skill32/3; 
						entFlare.alpha = ((entFlare.LfpTemp1)*(entFlare.LfpTemp1));
					}
					else { 
						entFlare.alpha = 0;
					}
					break;
				case 5:entFlare.alpha = 0.25 * ent.LfpVisAlpha * (ent.skill21/100);	break;										
				case 6:entFlare.alpha = ent.skill21-60; break;
				case 7:
				case 9:
					entFlare.LfpMaxAlpha = abs(ent.LfpPosX)/9;
					entFlare.alpha=(entFlare.LfpMaxAlpha/100) * ent.LfpVisAlpha * (ent.skill21/100);				
					break;										   
			}
		}
		/* NEONRHRE 2 */
		else if(subId == LfpNeonPipe2)
		{
			switch(index)
			{
				case 0:entFlare.alpha= 0.18 * (ent.skill21/100) * ent.LfpVisAlpha;break; 
				case 1:
					if (ent.skill31 > 0){
						entFlare.alpha=(1 * (ent.skill21/280) * (ent.skill31 + ent.skill31 + ent.skill31)) * (ent.LfpVisAlpha/100);
					}
					else {
						entFlare.alpha = 0;
					}	
					
					entFlare.LfpTemp1 = entFlare.alpha;						
					ent.skill31 = (-30 + ent.skill21);						
					break; 		
				case 2:
					if (ent.skill31 > 0) { 
						entFlare.alpha=(1 * (ent.skill21/280) * (ent.skill31)) * (ent.LfpVisAlpha/100);
					}
					else {
						entFlare.alpha = 0;
					}	
					
					entFlare.LfpTemp1 = entFlare.alpha;
					break;
				case 3:
					if (ent.skill31 > 0) {
						entFlare.alpha=(1 * (ent.skill21/280) * (ent.skill31)) * (ent.LfpVisAlpha/100);
					}						
					else {
						entFlare.alpha = 0;
					}	
					
					entFlare.LfpTemp1 = entFlare.alpha;
					break;
				case 4:
					if (ent.skill32 > 0) 
					{
						entFlare.LfpTemp1 = ent.skill32 / 5; 
						entFlare.alpha = ((entFlare.LfpTemp1 + entFlare.LfpTemp1) * (entFlare.LfpTemp1 + entFlare.LfpTemp1));
					} 
					else {entFlare.alpha = 0;}
					break;
			}	
		}
	}
	else if(mainId == LfpLTypePoint)
	{
		/* SONNE 1 */ 
		if(subId == LfpSun1)
		{
			switch(index)
			{
				case 0: 
				case 1: 
					entFlare.alpha = 0.6 * ent.LfpVisAlpha + random((0.02) * ent.LfpVisAlpha);
					break;
				case 2:
					entFlare.alpha = (ent.skill66/100) * ent.LfpVisAlpha;			
					break;
					
				case 3: LfpCtrlLightsGrpPointAdd2(ent,entFlare,index, 0.15, 0.008);break;
				case 4: LfpCtrlLightsGrpPointAdd2(ent,entFlare,index, 0.25, 0.01);break;
				case 5: LfpCtrlLightsGrpPointAdd2(ent,entFlare,index, 0.6, 0.01);break;
				case 6: LfpCtrlLightsGrpPointAdd2(ent,entFlare,index, 0.6, 0.01);break;
				case 7: LfpCtrlLightsGrpPointAdd2(ent,entFlare,index, 0.7, 1);break;
				case 8: LfpCtrlLightsGrpPointAdd2(ent,entFlare,index, 0.7, 1);break;
				case 9: LfpCtrlLightsGrpPointAdd2(ent,entFlare,index, 0.1, 0.008);break;
				case 10: LfpCtrlLightsGrpPointAdd2(ent,entFlare,index, 0.08, 0.008);break;
				case 11: LfpCtrlLightsGrpPointAdd2(ent,entFlare,index, 0.12, 0.02);break;
				case 12: LfpCtrlLightsGrpPointAdd2(ent,entFlare,index, 0.06, 0.008);break;
				case 13: LfpCtrlLightsGrpPointAdd2(ent,entFlare,index, 0.09, 0.008);break;
				case 14: LfpCtrlLightsGrpPointAdd2(ent,entFlare,index, 0.15, 0.008);break;
				case 15: LfpCtrlLightsGrpPointAdd2(ent,entFlare,index, 0.07, 0.008);break;
				case 16: LfpCtrlLightsGrpPointAdd2(ent,entFlare,index, 0.28, 0.01);break;
				case 17: LfpCtrlLightsGrpPointAdd2(ent,entFlare,index, 0.08, 0.008);break;
				case 18: LfpCtrlLightsGrpPointAdd2(ent,entFlare,index, 0.11, 0.01);break;
				case 19: LfpCtrlLightsGrpPointAdd2(ent,entFlare,index, 0.09, 0.008);break;
				case 20: LfpCtrlLightsGrpPointAdd2(ent,entFlare,index, 0.03, 0.008);break;
				case 21: LfpCtrlLightsGrpPointAdd2(ent,entFlare,index, 0.05, 0.01);break;
				case 22:	LfpCtrlLightsGrpPointAdd2(ent,entFlare,index, 0.10, 0.02);break;
			}						
		}
		/* SONNE 2 */ 
		else if(subId == LfpSun2)
		{
			switch(index)
			{
				case 0: 
				case 1:
					entFlare.alpha=((0.8 * ent.LfpVisAlpha)+random(0.05) * ent.LfpVisAlpha) * (ent.skill21/100);
					break;					
				case 2:
						if (ent.skill21 > 0) {
							entFlare.alpha=(((ent.skill66/100) + random(0.05)) * ent.LfpVisAlpha) * (ent.skill21/100);
						}
						else { 
							entFlare.alpha = 0;
					}				
					
					entFlare.LfpTemp1 = entFlare.alpha;
					break;
				
				case 3: 
					LfpCtrlLightsGrpPointAdd1(ent,entFlare,index, 0.45, 0.08);
					break;
				case 4: 
					LfpCtrlLightsGrpPointAdd1(ent,entFlare,index, 0.45, 0.08); 
					
					var t1[3];				
					var t2[3];
				
					t1[0] = abs(ent.LfpPosX) - (abs(ent.LfpPosX)-abs(ent.LfpPosX));
					t1[1] = abs(ent.LfpPosY) - (abs(ent.LfpPosY)-abs(ent.LfpPosY));
					
					ent.skill95 = sqrt(t1[0]*t1[0] + t1[1]*t1[1]);
					t2[0] = (ent.LfpPosX - ent.LfpPosX);
					t2[1] = (ent.LfpPosY - ent.LfpPosY);	
					
					vec_set(ent.skill50, ent.LfpPosX);
					vec_sub(ent.skill50, t2[0]); 
					vec_to_angle(ent.skill55, ent.skill50);	
					
					break;	
				case 5: LfpCtrlLightsGrpPointAdd1(ent,entFlare,index, 0.2,  0.08);break;
				case 6: LfpCtrlLightsGrpPointAdd1(ent,entFlare,index, 0.30, 0.08);break;
				case 7: LfpCtrlLightsGrpPointAdd1(ent,entFlare,index, 0.25, 0.08);break;
				case 8: LfpCtrlLightsGrpPointAdd1(ent,entFlare,index, 0.30, 0.08);break;
				case 9: LfpCtrlLightsGrpPointAdd1(ent,entFlare,index, 0.40, 0.08);break;
				case 10: LfpCtrlLightsGrpPointAdd1(ent,entFlare,index, 0.30, 0.08);break;
				case 11: LfpCtrlLightsGrpPointAdd1(ent,entFlare,index, 0.28, 0.08);break;
				case 12: LfpCtrlLightsGrpPointAdd1(ent,entFlare,index, 0.15, 0.08);break;
				case 13: LfpCtrlLightsGrpPointAdd1(ent,entFlare,index, 0.08, 0.08);break;
				case 14: LfpCtrlLightsGrpPointAdd1(ent,entFlare,index, 0.50, 0.08);break;
				case 15: LfpCtrlLightsGrpPointAdd1(ent,entFlare,index, 0.25, 0.08);break;
				case 16: LfpCtrlLightsGrpPointAdd1(ent,entFlare,index, 0.10, 0.08);break;
				case 17: LfpCtrlLightsGrpPointAdd1(ent,entFlare,index, 0.12, 0.08);break;
				case 18: LfpCtrlLightsGrpPointAdd1(ent,entFlare,index, 0.25, 0.08);break;
				case 19: LfpCtrlLightsGrpPointAdd1(ent,entFlare,index, 0.40, 0.08);break;
				case 20: LfpCtrlLightsGrpPointAdd1(ent,entFlare,index, 0.60, 0.08);break;
				case 21: LfpCtrlLightsGrpPointAdd1(ent,entFlare,index, 0.15, 0.08);break;
				case 22: LfpCtrlLightsGrpPointAdd1(ent,entFlare,index, 0.30, 0.08);break;
				case 23: LfpCtrlLightsGrpPointAdd1(ent,entFlare,index, 0.20, 0.08);break;
				case 24: LfpCtrlLightsGrpPointAdd1(ent,entFlare,index, 0.10, 0.08);break;
				case 25: LfpCtrlLightsGrpPointAdd1(ent,entFlare,index, 0.15, 0.05);break;
				case 26: LfpCtrlLightsGrpPointAdd1(ent,entFlare,index, 0.50, 0.08);break;								
			}						
		}
		/* STEHLAMPE */
		else if(subId == LfpLamp1)
		{
			switch(index)
			{
				case 0: entFlare.alpha= 0.3 * (ent.skill21/100) * ent.skill90; break; 
				case 1: entFlare.alpha= 1 * (ent.skill21/100) * ent.skill90; break; 		
				case 2:
					entFlare.alpha = ((1 * (ent.skill21/250) * (ent.skill31 * 3)) * ent.skill30) * (ent.LfpVisAlpha / 100);
					entFlare.LfpTemp1 = entFlare.alpha;						
					ent.skill31 = (-60 + ent.skill21);	
					break;
				case 3:entFlare.alpha= 0.15 * (ent.skill21/100) * ent.LfpVisAlpha; break;
				case 4:entFlare.alpha= 0.25 * (ent.skill21/100) * ent.LfpVisAlpha; break;
				case 5:entFlare.alpha= 0.5 * (ent.skill21/100) * ent.LfpVisAlpha;	break;	
				case 6:entFlare.alpha= 0.2 * (ent.skill21/100) * ent.LfpVisAlpha;	break; 		
				case 7:entFlare.alpha= 0.1 * (ent.skill21/100) * ent.LfpVisAlpha;	break;
				case 8:entFlare.alpha= 0.18 * (ent.skill21/100) * ent.LfpVisAlpha; break;
				case 9:
					entFlare.alpha = ((1 * (ent.skill21/1200) * (ent.skill31 + ent.skill31 + ent.skill31)) * ent.skill30) * (ent.LfpVisAlpha/100);
					entFlare.LfpTemp1 = entFlare.alpha;					
					break;		
			}					
		}
	}
	else if(mainId == LfpLTypeWindow)
	{
		
	}
	else
	{
		
	}
}


///
// Controls the visibility and effects of the lens flares
//
// <light> the light detail struct
// <blendOut> true if the lens flares are hidden, otherwise false

void LfpControlAllLightFlareEffects(LfpLight* light, bool blendOut)
{
	int mainId = light.MainGrpId;
	int subId = light.SubGrpId;
	int i;
	ENTITY* ent = light.Ent;
	
	if(blendOut)
	{
		ent.LfpVisAlpha = maxv(-1, ent.LfpVisAlpha - 40 * time_step);
		if(ent.LfpVisAlpha <= 0)
		{
			if(!light.IsClipped) 
			{
				light.IsClipped = true;
				
				if (light.Df)
				{
					if(light.Df.Flares)
					{
						for(i=0; i < light.Df.Count; i++)
						{
							you = ptr_for_handle((light.Df.Flares)[i]);
							if(you)
							{
								you.flags2 &= ~SHOW;
								you.alpha = 0;
							}
						}					
					}
				
					if(light.Df.Aura)
					{
						you = ptr_for_handle(light.Df.Aura);
						if(you)
						{
							you.flags2 &= ~SHOW;
							you.alpha = 0;						
						}						
					}
				}
		
				if (light.Sf)
				{
					if(light.Sf.Flares)
					{
						for(i=0; i < light.Sf.Count; i++)
						{
							you = ptr_for_handle((light.Sf.Flares)[i]);
							if(you)
							{
								you.flags2 &= ~SHOW;
								you.alpha = 0;
							}
						}					
					}
				
					if(light.Sf.Aura)
					{
						you = ptr_for_handle(light.Sf.Aura);
						if(you)
						{
							you.flags2 &= ~SHOW;
							you.alpha = 0;						
						}						
					}
				}
			}
			
			return;
		}
	}
	else
	{
		ent.LfpVisAlpha = minv(ent.LfpMaxLightAlpha, ent.LfpVisAlpha + 40 * time_step);		
	}
	
	light.IsClipped = false;

	
	if (light.Sf)
	{
		if(light.Sf.Flares)
		{
			for(i=0; i < light.Sf.Count;i++)
			{
				you = ptr_for_handle((light.Sf.Flares)[i]);
				if(you)
				{
					if(!blendOut)
					{
						LfpPlaceLightFlare(ent, you, mainId,subId,i);
						LfpControlLightFlare(ent, you, mainId, subId,i);
					}
						
					LfpAlphaLightFlare(ent, you, mainId, subId,i);
				}
			}
		}
	}
			
	LfpCtrlDirtyFlares(light);	
	
	
	
	if (abs(ent.LfpPosX) > abs(ent.LfpPosY)) {	ent.skill50 = abs(ent.LfpPosX/30);	}	
	if (abs(ent.LfpPosY) > abs(ent.LfpPosX)) {	ent.skill50 = abs(ent.LfpPosY/30);	}
	
	
	/* Let the crescent moon flare around the center of the light source*/
	var t1[3];
	t1[0] = (ent.LfpPosX - ent.LfpPosX);
	t1[1] = (ent.LfpPosY - ent.LfpPosY);
	t1[2] = 0;
	
	vec_set(ent.skill40, ent.LfpPosX);
	vec_sub(ent.skill40, t1); 
	vec_to_angle(ent.skill55, ent.skill40);
	
	
	if(mainId == LfpLTypeLong && subId == LfpNeonPipe1)
	{
		ent.skill32 = (-50+ent.skill21);		
	}
	
	if(mainId == LfpLTypeLong && subId == LfpNeonPipe2)
	{	
		ent.skill32 = (-80+ent.skill21);	
	}
}




///
// Aura
// ------------
// 
/// 


///
// Controls the behavior of the auras for a light model
//
// <light> the light detail struct
void LfpControlLightAuras(LfpLight* light)
{
	if(light.Aura == NULL) { return; }
	
	int mainId = light.MainGrpId;	
	int subId = light.SubGrpId;
			
	ENTITY* ent = light.Ent;
	you = ptr_for_handle(light.Aura);
	if(you)
	{
		you.tilt = ent.tilt;
		you.alpha = 10+((ent.LfpMaxLightAlpha/100) * (ent.skill21/100) * ent.LfpVisAlpha);
		you.alpha=minv(50, you.alpha);
		
		if(mainId == LfpLTypeLong)
		{

		}
		else if(mainId == LfpLTypePoint)
		{
			you.pan = ent.pan + 0.01;		
		}
		else if(mainId == LfpLTypeWindow)
		{
	
		}
		else
		{
			
		}		
	}
}

///
// Creates a view aura at the position of the light (screen)
//
// <light> the light detail struct
// <ent> the light model
// <layer> view layer
// <name> Name of the aura file
var LfpCreateLightAura(LfpLight* light,var layer, char* name)
{
	ENTITY* ent = light.Ent;
	you = ent_createlayer(name,0,layer);
	
	#ifdef SSAO
		you.client_id = viewSsaoCombine;
		setSsaoSurface(you, SSAO_TYPE_SOFTALPHA);			
	#endif
	
	your.LfpPivotDist = 1;
	vec_set(your.blue, vector(ent.LfpBlue,ent.LfpGreen,ent.LfpRed));
	
	your.flags |= PASSABLE|BRIGHT|TRANSLUCENT|NOFOG;
	your.ambient = 100;
	your.albedo = -100;
	your.alpha = 0;

	return handle(you);
}

///
// Creates a 3D aura on the light model.
//
// <light> the light detail struct
// <ent> the light model
var LfpCreateLightAura(LfpLight* light)
{
	int mainId = light.MainGrpId;
	int subId = light.SubGrpId;
	ENTITY* ent = light.Ent;
	ENTITY* ret = NULL;
	static const char* lf10Grp9 = "LF_10grp9.tga";
	static const char* lfSunB = "LF_sunB.tga";
	
	if(mainId == LfpLTypeLong)
	{
		switch(subId)
		{
			case LfpNeonPipe1:
			case LfpNeonPipe2:
		
				ret = ent_create(lf10Grp9, ent.x, NULL);			
				vec_set(ret.scale_x, vector(1,0.2,0));
				break;	
		}		
	}
	else if(mainId == LfpLTypePoint)
	{
		switch(subId)
		{
			case LfpSun1:
				break;
			case LfpSun2:
				ret = ent_create(lfSunB, ent.x, NULL);
				ret.pan = ent.pan+0.1;
				break;
			case LfpLamp1:	break;
		}
	}
	else if(mainId == LfpLTypeWindow)
	{

	}
	else
	{
		
	}
	
	if(ret)
	{		
		LfpSetLightAuraMaterial(light);	
		
		if(light.MatAura)
		{
			ret.material = light.MatAura;			
		}

		ret.flags |= PASSABLE|BRIGHT|TRANSLUCENT|NOFOG;
		ret.ambient = 100;
		ret.albedo = -100;
		ret.alpha = 100;
			
		#ifdef SSAO
			setSsaoSurface(ret, SSAO_TYPE_SOFTALPHA);	
		#endif
				
			
		return handle(ret);
		//DbgWriteLog("\nLight aura of the group %s-%s-Nr. %d created.", LfpGetGrpText(mainId, false), LfpGetGrpText(subId, true), (long)ent.LfpOrderId);			
	}
	

	return NULL;
}

///
// Sets the material for an aura.
//
// <light> the light detail struct
void LfpSetLightAuraMaterial(LfpLight* light)
{
	int mainId = light.MainGrpId;
	int subId = light.SubGrpId;
	ENTITY* ent = light.Ent;
	
	if(mainId == LfpLTypeLong)
	{
		switch(subId)
		{
			case LfpNeonPipe1:
			case LfpNeonPipe2:
			
				if(ent.LfpOrderId == 1) { light.MatAura = LfpMatCreate(10,NULL, vector(100,100,250) ,vector(50,50,250));}
				if(ent.LfpOrderId == 2) { light.MatAura = LfpMatCreate(10,NULL, vector(250,100,100) ,vector(250,50,50));}
				if(ent.LfpOrderId == 3) { light.MatAura = LfpMatCreate(10,NULL, vector(100,250,100) ,vector(50,250,50));}
				break;
					
		}		
	}
	else if(mainId == LfpLTypePoint)
	{
		switch(subId)
		{
			case LfpSun1:break;
			case LfpSun2:break;
			case LfpLamp1:	break;
		}
	}
	else if(mainId == LfpLTypeWindow)
	{

	}
	else
	{
		
	}
	
		
}










///
// Lights
// ------------
// 
/// 

///
// Cross fade from a certain distance to the source
//
// <light> the light detail struct
// <dist> distance to the source, but which is blended
void LfpControlLightBlending(LfpLight* light, var dist)
{
	ENTITY* ent = light.Ent;
	
	/* The camera points exactly into the light source and fades in. */
	ent.skill67 = abs(ent.LfpPosX);
	ent.skill68 = abs(ent.LfpPosY);
	
	ent.skill67 = ifelse(ent.skill67 == 0, 0.1, ent.skill67);
	ent.skill67 = ifelse(ent.skill65 == 0, 0.1, ent.skill67);
		
	if (ent.skill67 > ent.skill68)
	{
		ent.skill66 = (100-ent.skill67)*(ent.skill65/100);	
	}
	
	if (ent.skill68 > ent.skill67) 
	{
		ent.skill66 = (100-ent.skill68)*(ent.skill65/100);
	}

	if(dist > 0)
	{
		if (vec_dist(ent.x, camera.x) < dist+1) 
		{
			ent.skill65 = dist-vec_dist(ent.x, camera.x);
			ent.skill65 = minv(100,ent.skill65);
		} 
		else 
		{
			ent.skill65 = 1;
		}			
	}	
}

///
// Controls the alpha values of individual light models
// This function is part of the overall control logic
//
// <light> the light detail struct
// <dist> Distance from camera to light model
void LfpControlLights(LfpLight* light, var dist)
{
	int mainId = light.MainGrpId;
	int subId = light.SubGrpId;	
	ENTITY* ent = light.Ent;

	ent.skill21 = 100-((100 / ent.LfpLightDist) * dist);	

	if(mainId == LfpLTypeLong)
	{
		switch(subId)
		{
			case LfpNeonPipe1:
			case LfpNeonPipe2:
		
				break;
		}
	}
	else if(mainId == LfpLTypePoint)
	{
		if(light.Df)
		{
			ent.skill24 = 90-((100/ent.LfpLightDist) * dist);
			ent.skill23 = ((100/ent.LfpMaxLightAlpha) + random(0.3)) * ent.LfpVisAlpha;
			ent.skill22 = ((ent.LfpMaxLightAlpha/100) * ent.skill24) * (ent.skill23/100) * (ent.skill24/100) * (ent.skill26/100); 
	
			if (dist < 301) 
			{
				ent.skill26 = (100/250) * (dist-100);
				
				if(ent.skill26 > 0)
					ent.skill27 = 100/ent.skill26;
				else
					ent.skill27 = 1;
			} 
			else
			{
				ent.skill26 = 100;
				ent.skill27 = 1;
			}
			
			light.Radius = (light.InitialRadius / 100) * (ent.skill24 * ent.skill27);
		}
		
		if(subId == LfpSun1)
		{
			ent.skill66 = 20;
			ent.skill21 = 100;

			vec_set(ent.skill80, camera.x);		
			vec_sub(ent.skill80, ent.x);
			vec_to_angle(ent.pan, ent.skill80);
			
			ent.alpha = (ent.LfpMaxLightAlpha - (ent.LfpVisAlpha / 2)) + random(0.02 * ent.LfpVisAlpha);
	
			ent.skill23 = (100/ent.LfpMaxLightAlpha) * ent.LfpVisAlpha;
			ent.skill22 = ((ent.LfpMaxLightAlpha / 100) * ent.skill21) * (ent.skill23/100);
		}
		else if(subId == LfpSun2)
		{
			ent.skill21 = 100-(((100/ent.LfpLightDist) * dist) / 0.5);
		}
		else if(subId == LfpLamp1)	
		{
			// Default
		}
	}
	else if(mainId == LfpLTypeWindow)
	{
		ent.alpha = ent.skill25;
		
		ent.skill25 = (ent.LfpMaxLightAlpha / 100) * ent.skill21;
		ent.skill24 = 100-((100/ent.LfpLightDist) * dist);
		ent.skill23 = (100/ent.LfpMaxLightAlpha) * ent.LfpVisAlpha;
		
		ent.skill22 = ((ent.LfpMaxLightAlpha/100) * ent.skill24) * (ent.skill23/100) * (ent.skill24/100) * (ent.skill26/100);
		
		if (dist < 301) 
		{
			ent.skill26 = 0.4 * (dist - 100);
			ent.skill27 = 100 / ent.skill26;
		} 
		else 
		{
			ent.skill26 = 100;
			ent.skill27 = 1;
		}
		
		light.Radius = (light.InitialRadius / 100) * (ent.skill24 * ent.skill27);		
	}
	else
	{
		
	}
}

///
// Sets the material for a light model.
//
// <light> the light detail struct
void LfpSetLightMaterial(LfpLight* light)
{
	int mainId = light.MainGrpId;
	int subId = light.SubGrpId;
	ENTITY* ent = light.Ent;
	
	if(mainId == LfpLTypeLong)
	{
		switch(subId)
		{
			case LfpNeonPipe1:
			case LfpNeonPipe2:
				if(ent.LfpOrderId == 1) { ent.material = LfpMatCreate(10,NULL, vector(100,100,250), vector(50,50,250)); }	
				if(ent.LfpOrderId == 2) { ent.material = LfpMatCreate(10,NULL, vector(250,100,100), vector(250,50,50)); }
				if(ent.LfpOrderId == 3) { ent.material = LfpMatCreate(10,NULL, vector(100,250,100), vector(50,250,50)); }									
				break;
		}
	}
	else if(mainId == LfpLTypePoint)
	{
		switch(subId)
		{
			case LfpSun1:break;	
			case LfpSun2:break;
			case LfpLamp1:break;
		}
	}
	else if(mainId == LfpLTypeWindow)
	{
	}
	else
	{
		
	}


	//DbgWriteLog("\nMaterial assigned to group %s-%s", LfpGetGrpText(mainId, false), LfpGetGrpText(subId, true));	
}

///
// Sets the parent of the light model, for example if the light gets a more complex model as a basic framework
//
// <light> the light detail struct
void LfpSetLightParent(LfpLight* light)
{
	int mainId = light.MainGrpId;	
	int subId = light.SubGrpId;
	static const char* lampeB = "lampeB.mdl";
	//static const char* lampe2 = "lampe2.mdl";
	
	ENTITY* ent = light.Ent;
	you = NULL;
	
	if(mainId == LfpLTypeLong)
	{
		switch(subId)
		{
			case LfpNeonPipe1:
			case LfpNeonPipe2:
				you = ent_create(lampeB, ent.x, NULL);
				vec_set(you.scale_x, ent.scale_x);								
				break;				
		}
	}
	else if(mainId == LfpLTypePoint)
	{
		switch(subId)
		{
			case LfpSun1:break;
			case LfpSun2:break;
			case LfpLamp1:
				//you = ent_create(lampe2, vector(ent.x, ent.y, ent.z+40), NULL);				
				break;
		}
	}
	else if(mainId == LfpLTypeWindow)
	{
	}
	else
	{
		
	}
	
	if(you) 
	{
		ent.parent = you;
		DbgWriteLog("\nDem Lichtmodel der Gruppe %s-%s wurde ein Parent zugewiesen", LfpGetGrpText(mainId, false), LfpGetGrpText(subId, true));
	}
		 
	
	//DbgWriteLog("\nMaterial assigned to group %s-%s", LfpGetGrpText(mainId, false), LfpGetGrpText(subId, true));	
}

///
// Deletes the resources of the current light model
//
// <light> the light detail struct
void LfpDeleteLightResources(LfpLight* light)
{
	if(light.IsCreated)
	{	
		LfpDeleteLightFlares(light);
		LfpDeleteDirtyFlares(light);
		
		if(light.Aura)
		{		
			you = ptr_for_handle(light.Aura);
			if(you)
			{
				if(light.MatAura)
				{
					ptr_remove(light.MatAura);
					light.MatAura = NULL;
				}
				
				ent_purge(you);
				ptr_remove(you);
				you = NULL;
				light.Aura = NULL;
			}
		}
		
		light.IsCreated = false;
		
//		int mainId = light.MainGrpId;	
//		int subId = light.SubGrpId;	
//		
		//DbgWriteLog("\nLens flares of the light group \"%s-%s-Nr. %d\" removed", LfpGetGrpText(mainId, false), LfpGetGrpText(subId,true), (long)light.Ent.LfpOrderId);	
	} 
}

///
// Generates a light when the range limit has been reached
//
// <light> the light detail struct
void LfpCreateLight(LfpLight* light)
{
	if(!light.IsCreated)
	{
		ENTITY* ent = light.Ent;
		ent.flags |= PASSABLE|BRIGHT|TRANSLUCENT;	
		ent.ambient = 100;
		ent.albedo = -100;	
		ent.alpha = ent.LfpVisAlpha = ent.LfpMaxLightAlpha;
				
		/* Initial values if they have not been set in WED */
		if(ent.LfpLightDist <= 0) { ent.LfpLightDist = 500;}
		if(ent.LfpRadius <= 0) { ent.LfpRadius = 1200;}	
		if(ent.LfpMaxDustAlpha <= 0) { ent.LfpMaxDustAlpha = 70;}	
		if(ent.LfpMaxLightAlpha <= 0) { ent.LfpMaxLightAlpha = 70;}
		if(ent.LfpMaxAuraAlpha <= 0) { ent.LfpMaxAuraAlpha = 50;}	

		light.InitialRadius = ent.LfpRadius;
		light.Radius = ent.LfpRadius;
		
		int mainId = light.MainGrpId;	
		int subId = light.SubGrpId;		
		
		/* Activates dirty flares if flag 8 (LfpHasDirtyFlares) is active */
		if(light.Df)
		{
			static const char* lfDirtF = "LF_dirtF.tga";
			
			/* Dirty flares have their own aura */
			light.Df.Aura = LfpCreateLightAura(light,5, lfDirtF);
			light.Df.Flares = LfpCreateDirtyFlares(light);
		}
		
		light.Aura = LfpCreateLightAura(light);
		light.Sf.Flares = LfpCreateLightFlares(light);
		
		light.IsCreated = true;
	}
}




